// -*- Java -*-
/*!
 * @file  RSNPCameraImpl.java
 * @brief Connect_RSNPServer
 * @date  $Date$
 *
 * @author Author: Hiroshi Ogiya
 * Contact Information: Shibaura Insutitute of
 * Technology. 3-7-5 Toyosu Koto-ku Tokyo 135-8548
 * Email:md14017@shibaura-it.ac.jp
 *
 * Apply the correction BSD license.
 *
 * $Id$
 */

import rsnp.RSNPAccess;
import rsnp.RSNPRecivedData;
import rtm.RTMRecivedData;
import tool.XML_Tools;
import RTC.CameraImage;
import RTC.PanTiltAngles;
import RTC.TimedLong;
import jp.go.aist.rtm.RTC.DataFlowComponentBase;
import jp.go.aist.rtm.RTC.Manager;
import jp.go.aist.rtm.RTC.port.InPort;
import jp.go.aist.rtm.RTC.port.OutPort;
import jp.go.aist.rtm.RTC.util.DataRef;
import RTC.ReturnCode_t;

/*!
 * @class RSNPCameraImpl
 * @brief Connect_RSNPServer
 *
 * Receive the Camera Images from another component and send them
 * to the RSNP Server using distribute_senser_info method
 * (Multimedia_Profile class).
 * Recceive the angle data from the RSNP using data_push method
 * (Push_Data class) Server and send each of them.
 *
 * Input: CameraImage
 * Output: PanTiltAngles, PanTiltAngles
 *
 * Robot Service Network Protocol2.2 Specification
 *
 */
public class RSNPCameraImpl extends DataFlowComponentBase {

  /*!
   * @brief constructor
   * @param manager Maneger Object
   */
	public RSNPCameraImpl(Manager manager) {  
        super(manager);
        // <rtc-template block="initializer">
        m_CameraImage_val = new CameraImage();
        m_CameraImage = new DataRef<CameraImage>(m_CameraImage_val);
        m_CameraImageIn = new InPort<CameraImage>("CameraImage", m_CameraImage);
        m_CameraControlPan_val = new PanTiltAngles();
        m_CameraControlPan = new DataRef<PanTiltAngles>(m_CameraControlPan_val);
        m_CameraControlPanOut = new OutPort<PanTiltAngles>("CameraControlPan", m_CameraControlPan);
        m_CameraControlTilt_val = new PanTiltAngles();
        m_CameraControlTilt = new DataRef<PanTiltAngles>(m_CameraControlTilt_val);
        m_CameraControlTiltOut = new OutPort<PanTiltAngles>("CameraControlTilt", m_CameraControlTilt);
        // </rtc-template>

    }

    /**
     *
     * The initialize action (on CREATED->ALIVE transition)
     * formaer rtc_init_entry() 
     *
     * @return RTC::ReturnCode_t
     * 
     * 
     */
    @Override
    protected ReturnCode_t onInitialize() {
        // Registration: InPort/OutPort/Service
        // <rtc-template block="registration">
        // Set InPort buffers
        addInPort("CameraImage", m_CameraImageIn);
        
        // Set OutPort buffer
        addOutPort("CameraControlPan", m_CameraControlPanOut);
        addOutPort("CameraControlTilt", m_CameraControlTiltOut);
        // </rtc-template>
        return super.onInitialize();
    }

    /***
     *
     * The finalize action (on ALIVE->END transition)
     * formaer rtc_exiting_entry()
     *
     * @return RTC::ReturnCode_t
     * 
     * 
     */
//    @Override
//    protected ReturnCode_t onFinalize() {
//        return super.onFinalize();
//    }

    /***
     *
     * The startup action when ExecutionContext startup
     * former rtc_starting_entry()
     *
     * @param ec_id target ExecutionContext Id
     *
     * @return RTC::ReturnCode_t
     * 
     * 
     */
//    @Override
//    protected ReturnCode_t onStartup(int ec_id) {
//        return super.onStartup(ec_id);
//    }

    /***
     *
     * The shutdown action when ExecutionContext stop
     * former rtc_stopping_entry()
     *
     * @param ec_id target ExecutionContext Id
     *
     * @return RTC::ReturnCode_t
     * 
     * 
     */
//    @Override
//    protected ReturnCode_t onShutdown(int ec_id) {
//        return super.onShutdown(ec_id);
//    }

    /***
     *
     * The activated action (Active state entry action)
     * former rtc_active_entry()
     *
     * @param ec_id target ExecutionContext Id
     *
     * @return RTC::ReturnCode_t
     * 
     * 
     */
    @Override
    protected ReturnCode_t onActivated(int ec_id) {
    	RSNPAccess rsnp = RSNPAccess.getInstance();
    	rsnp.open();
    	
        return super.onActivated(ec_id);
    }

    /***
     *
     * The deactivated action (Active state exit action)
     * former rtc_active_exit()
     *
     * @param ec_id target ExecutionContext Id
     *
     * @return RTC::ReturnCode_t
     * 
     * 
     */
    @Override
    protected ReturnCode_t onDeactivated(int ec_id) {
    	RSNPAccess rsnp = RSNPAccess.getInstance();
        rsnp.close();
        
        return super.onDeactivated(ec_id);
    }

    /***
     *
     * The execution action that is invoked periodically
     * former rtc_active_do()
     *
     * @param ec_id target ExecutionContext Id
     *
     * @return RTC::ReturnCode_t
     * 
     * 
     */
    @Override
    protected ReturnCode_t onExecute(int ec_id) {
    	if(m_CameraImageIn.isNew()){
    		RTMRecivedData rtmrecive = RTMRecivedData.getInstance();
    		m_CameraImageIn.read();
    		byte[] buffer = m_CameraImage.v.pixels;
    		rtmrecive.set_camera_image(buffer);
    	}
    	String command = RSNPRecivedData.getInstance().get_recive_data("camera_control");
    	if(command != null){
    		if(command.equals("start")){
    			//System.out.println(RSNPAccess.getInstance().push_data());
    			
    			XML_Tools xml= new XML_Tools(RSNPAccess.getInstance().push_data());
    			int pitch = Integer.parseInt(xml.get_target_text("data", "pitch"));
    			int yaw = Integer.parseInt(xml.get_target_text("data", "yaw"));
    			System.out.println("pitch:"+pitch+",yaw:"+yaw);
    			TimedLong l_pitch = new TimedLong();
    			TimedLong l_yaw = new TimedLong();
    			l_pitch.tm = new RTC.Time(0,0);
    			l_yaw.tm = new RTC.Time(0,0);
    			l_pitch.data = pitch;
    			l_yaw.data = yaw;
    			m_CameraControlPanOut.write();
    			m_CameraControlTiltOut.write();
    		}
    		
    	}
    	
        return super.onExecute(ec_id);
    }

    /***
     *
     * The aborting action when main logic error occurred.
     * former rtc_aborting_entry()
     *
     * @param ec_id target ExecutionContext Id
     *
     * @return RTC::ReturnCode_t
     * 
     * 
     */
//  @Override
//  public ReturnCode_t onAborting(int ec_id) {
//      return super.onAborting(ec_id);
//  }

    /***
     *
     * The error action in ERROR state
     * former rtc_error_do()
     *
     * @param ec_id target ExecutionContext Id
     *
     * @return RTC::ReturnCode_t
     * 
     * 
     */
    @Override
    public ReturnCode_t onError(int ec_id) {
        return super.onError(ec_id);
    }

    /***
     *
     * The reset action that is invoked resetting
     * This is same but different the former rtc_init_entry()
     *
     * @param ec_id target ExecutionContext Id
     *
     * @return RTC::ReturnCode_t
     * 
     * 
     */
    @Override
    protected ReturnCode_t onReset(int ec_id) {
        return super.onReset(ec_id);
    }

    /***
     *
     * The state update action that is invoked after onExecute() action
     * no corresponding operation exists in OpenRTm-aist-0.2.0
     *
     * @param ec_id target ExecutionContext Id
     *
     * @return RTC::ReturnCode_t
     * 
     * 
     */
//    @Override
//    protected ReturnCode_t onStateUpdate(int ec_id) {
//        return super.onStateUpdate(ec_id);
//    }

    /***
     *
     * The action that is invoked when execution context's rate is changed
     * no corresponding operation exists in OpenRTm-aist-0.2.0
     *
     * @param ec_id target ExecutionContext Id
     *
     * @return RTC::ReturnCode_t
     * 
     * 
     */
//    @Override
//    protected ReturnCode_t onRateChanged(int ec_id) {
//        return super.onRateChanged(ec_id);
//    }
//
    // DataInPort declaration
    // <rtc-template block="inport_declare">
    protected CameraImage m_CameraImage_val;
    protected DataRef<CameraImage> m_CameraImage;
    /*!
     * Get the Camera Image.
     * - Type: CameraImage
     * - Number: 1
     * - Frequency: 1000.0
     * - Operation Cycle: 1000.0
     */
    protected InPort<CameraImage> m_CameraImageIn;

    
    // </rtc-template>

    // DataOutPort declaration
    // <rtc-template block="outport_declare">
    protected PanTiltAngles m_CameraControlPan_val;
    protected DataRef<PanTiltAngles> m_CameraControlPan;
    /*!
     * Send the attitude data(Pan) of the HMD.
     * - Type: PanTiltAngles
     * - Number: 1
     * - Unit: degree
     * - Frequency: 1000.0
     * - Operation Cycle: 1000.0
     */
    protected OutPort<PanTiltAngles> m_CameraControlPanOut;

    protected PanTiltAngles m_CameraControlTilt_val;
    protected DataRef<PanTiltAngles> m_CameraControlTilt;
    /*!
     * Send the attitude data(Tilt) of the HMD.
     * - Type: PanTiltAngles
     * - Number: 1
     * - Unit: degree
     * - Frequency: 1000.0
     * - Operation Cycle: 1000.0
     */
    protected OutPort<PanTiltAngles> m_CameraControlTiltOut;

    
    // </rtc-template>

    // CORBA Port declaration
    // <rtc-template block="corbaport_declare">
    
    // </rtc-template>

    // Service declaration
    // <rtc-template block="service_declare">
    
    // </rtc-template>

    // Consumer declaration
    // <rtc-template block="consumer_declare">
    
    // </rtc-template>


}
